<?php

namespace App\Http\Controllers\Admin\Keuangan;

use ZipArchive;
use App\Models\Guru;
use App\Models\Pegawai;
use App\Models\Kuitansi;
use App\Models\Pelatihan;
use App\Support\Terbilang;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Http\Controllers\Controller;

class KuitansiController extends Controller
{
    /* =========================
     * INDEX
     * ========================= */
    public function index(Request $request)
    {
        $search = $request->search;
        $tahun  = $request->tahun;

        $query = Kuitansi::with(['ptk', 'pelatihan'])->latest();

        if ($search) {
            $query->whereHasMorph(
                'ptk',
                [Guru::class, Pegawai::class],
                function ($q) use ($search) {
                    $q->where('nama_guru', 'like', "%{$search}%")
                        ->orWhere('nama', 'like', "%{$search}%")
                        ->orWhere('nip', 'like', "%{$search}%")
                        ->orWhere('nuptk', 'like', "%{$search}%");
                }
            );
        }

        if ($tahun) {
            $query->where('tahun_anggaran', $tahun);
        }

        $rows = $query->paginate(15)->withQueryString();

        return view('admin.keuangan.kuitansi.index', compact('rows', 'search', 'tahun'));
    }

    /* =========================
     * CREATE
     * ========================= */
    public function create()
    {
        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')
            ->get(['id', 'nama_pelatihan']);

        $selectedPtk = null;

        return view('admin.keuangan.kuitansi.create', compact(
            'pelatihanList',
            'selectedPtk'
        ));
    }

    /* =========================
     * STORE
     * ========================= */
    public function store(Request $request)
    {
        $data = $request->validate([
            'ptk' => ['required'],
            'pelatihan_id' => ['nullable', 'exists:pelatihan,id'],

            'tahun_anggaran' => ['required', 'string', 'max:4'],
            'bukti_kas_no' => ['nullable', 'string', 'max:100'],
            'beban' => ['nullable', 'string', 'max:150'],
            'akun' => ['nullable', 'string', 'max:150'],

            'sudah_terima_dari' => ['nullable', 'string', 'max:255'],
            'untuk_pembayaran' => ['required', 'string'],

            'honor_bruto' => ['nullable', 'numeric', 'min:0'],
            'pph21_rate' => ['nullable', 'numeric', 'min:0'],

            'transport_nominal' => ['nullable', 'numeric', 'min:0'],
            'transport_kali' => ['nullable', 'integer', 'min:0'],

            'tanggal_kuitansi' => ['nullable', 'date'],
        ]);

        /* === PTK === */
        [$ptkPrefix, $ptkId] = explode('_', $data['ptk']);
        $ptkType = $ptkPrefix === 'guru' ? Guru::class : Pegawai::class;

        /* === NILAI DASAR === */
        $honor = (int) round($data['honor_bruto'] ?? 0);
        $rate  = (float) ($data['pph21_rate'] ?? 0);

        $transportNominal = (int) ($data['transport_nominal'] ?? 0);
        $transportKali    = (int) ($data['transport_kali'] ?? 0);
        $transportTotal   = $transportNominal * $transportKali;

        /* === VALIDASI MINIMAL ADA ISI === */
        if ($honor === 0 && $transportTotal === 0) {
            return back()
                ->withErrors(['honor_bruto' => 'Honor atau transport wajib diisi salah satu'])
                ->withInput();
        }

        /* === HITUNGAN === */
        $pph        = (int) round($honor * ($rate / 100)); // PPH hanya dari honor
        $totalBruto = $honor + $transportTotal;
        $net        = $totalBruto - $pph;

        /* === TERBILANG DARI TOTAL === */
        $terbilang = trim(Terbilang::make($totalBruto));

        $kuitansi = Kuitansi::create([
            'ptk_type' => $ptkType,
            'ptk_id'   => $ptkId,

            'pelatihan_id' => $data['pelatihan_id'] ?? null,

            'tahun_anggaran' => $data['tahun_anggaran'],
            'bukti_kas_no' => $data['bukti_kas_no'] ?? null,
            'beban' => $data['beban'] ?? null,
            'akun' => $data['akun'] ?? null,

            'sudah_terima_dari' => $data['sudah_terima_dari'] ?? null,
            'untuk_pembayaran' => $data['untuk_pembayaran'],

            'transport_nominal' => $transportNominal,
            'transport_kali' => $transportKali,
            'transport_total' => $transportTotal,

            'honor_bruto' => $honor,
            'pph21_rate' => $rate,
            'pph21_amount' => $pph,
            'jumlah_diterima' => $net,

            'tanggal_kuitansi' => $data['tanggal_kuitansi'] ?? null,
            'terbilang' => $terbilang,
            'created_by' => auth()->id(),
        ]);

        return redirect()
            ->route('admin.keuangan.kuitansi.show', $kuitansi->id)
            ->with('success', 'Kuitansi berhasil dibuat.');
    }


    /* =========================
     * SHOW / PRINT
     * ========================= */
    public function show(Kuitansi $kuitansi)
    {
        $kuitansi->load(['ptk', 'pelatihan']);
        return view('admin.keuangan.kuitansi.show', compact('kuitansi'));
    }

    public function print(Kuitansi $kuitansi)
    {
        $kuitansi->load(['ptk', 'pelatihan']);
        return view('admin.keuangan.kuitansi.print', compact('kuitansi'));
    }

    /* =========================
     * DOWNLOAD
     * ========================= */
    public function download(Kuitansi $kuitansi)
    {
        $kuitansi->load(['ptk', 'pelatihan']);

        $kopPath = public_path('images/kop_bgtk_revisi.jpeg');
        $kop = file_exists($kopPath) ? $kopPath : null;

        $nama = $kuitansi->ptk->nama_guru
            ?? $kuitansi->ptk->nama
            ?? 'ptk';

        $pdf = Pdf::loadView('admin.keuangan.kuitansi.print', [
            'kuitansi' => $kuitansi,
            'kop' => $kop,
        ])->setPaper('a4', 'portrait');

        return $pdf->download("kuitansi_{$nama}_{$kuitansi->id}.pdf");
    }

    /* =========================
     * EDIT
     * ========================= */
    public function edit(Kuitansi $kuitansi)
    {
        $kuitansi->load(['ptk', 'pelatihan']);

        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')
            ->get(['id', 'nama_pelatihan']);

        // value select2
        $selectedPTK = null;
        if ($kuitansi->ptk) {
            $prefix = $kuitansi->ptk_type === Guru::class ? 'guru' : 'pegawai';
            $nama   = $kuitansi->ptk->nama_guru ?? $kuitansi->ptk->nama;
            $nip    = $kuitansi->ptk->nip ?? null;

            $selectedPTK = [
                'id'   => "{$prefix}_{$kuitansi->ptk_id}",
                'text' => $nama . ($nip ? " - ({$nip})" : ''),
            ];
        }

        return view('admin.keuangan.kuitansi.edit', compact(
            'kuitansi',
            'pelatihanList',
            'selectedPTK'
        ));
    }

    /* =========================
     * UPDATE
     * ========================= */
    public function update(Request $request, Kuitansi $kuitansi)
    {
        $data = $this->store($request)->getData();
        return redirect()
            ->route('admin.keuangan.kuitansi.show', $kuitansi->id)
            ->with('success', 'Kuitansi berhasil diperbarui.');
    }

    /* =========================
     * DESTROY
     * ========================= */
    public function destroy(Kuitansi $kuitansi)
    {
        $kuitansi->delete();

        return redirect()
            ->route('admin.keuangan.kuitansi.index')
            ->with('success', 'Kuitansi berhasil dihapus.');
    }

    public function downloadAll()
    {
        // 🔥 ANTI TIMEOUT
        ini_set('max_execution_time', 300); // 5 menit
        ini_set('memory_limit', '512M');

        $totalData = Kuitansi::count();

        if ($totalData === 0) {
            return back()->with('error', 'Tidak ada data kuitansi untuk diunduh.');
        }

        $tempDir = storage_path('app/temp_kuitansi_pdf/');
        if (!file_exists($tempDir)) {
            mkdir($tempDir, 0777, true);
        }

        // bersihkan file lama
        foreach (glob($tempDir . '*.pdf') as $oldFile) {
            @unlink($oldFile);
        }

        // 🔥 KOP LOAD SEKALI
        $kopPath = public_path('images/kop_bgtk_revisi.jpeg');
        $kop = file_exists($kopPath) ? $kopPath : null;

        // 🔥 PROSES BERTAHAP
        Kuitansi::with(['ptk', 'pelatihan'])
            ->orderBy('id')
            ->chunk(10, function ($rows) use ($tempDir, $kop) {

                foreach ($rows as $kuitansi) {

                    $rawName = $kuitansi->ptk?->nama_guru
                        ?? $kuitansi->ptk?->nama
                        ?? 'ptk';

                    $safeName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $rawName);
                    $filename = 'kuitansi_' . $safeName . '_' . $kuitansi->id . '.pdf';

                    Pdf::loadView('admin.keuangan.kuitansi.print', [
                        'kuitansi' => $kuitansi,
                        'kop'      => $kop,
                    ])
                        ->setPaper('a4', 'portrait')
                        ->setOptions([
                            'isRemoteEnabled' => true,
                            'dpi' => 96,
                            'defaultFont' => 'Times-Roman',
                        ])
                        ->save($tempDir . $filename);
                }
            });

        $pdfFiles = glob($tempDir . '*.pdf');

        if (count($pdfFiles) === 0) {
            return back()->with('error', 'PDF kuitansi tidak berhasil dibuat.');
        }

        // 🔥 ZIP
        $zipPath = storage_path('app/kuitansi_all.zip');
        $zip = new \ZipArchive;

        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE)) {
            foreach ($pdfFiles as $file) {
                $zip->addFile($file, basename($file));
            }
            $zip->close();
        }

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }
}
