<?php

namespace App\Http\Controllers\Admin;

use App\Models\Role;
use App\Models\User;
use App\Models\Pelatihan;
use Illuminate\Http\Request;
use App\Models\PelatihanKelas;
use App\Models\PelatihanPeserta;
use App\Models\RegistrasiPeserta;
use App\Http\Controllers\Controller;

class PelatihanKelasController extends Controller
{
    public function index(Pelatihan $pelatihan)
    {

        $kelas = PelatihanKelas::with(['adminLms', 'fasil'])
            ->withCount('peserta')
            ->where('pelatihan_id', $pelatihan->id)
            ->orderBy('kode_kelas')
            ->get();

        return view('admin.pelatihan.kelas.index', compact('pelatihan', 'kelas'));
    }

    public function show(Request $request, PelatihanKelas $kelas)
    {
        $kelas->load([
            'pelatihan',
            'adminLms.role',
            'fasil.role',
            'peserta.registrasi'
        ]);

        $role = $request->get('role', 'all');
        $data = collect();

        /**
         * ======================
         * ADMIN KELAS
         * ======================
         */
        if ($kelas->adminLms) {
            $data->push((object)[
                'id'   => $kelas->adminLms->id,
                'nama' => $kelas->adminLms->name,
                'unit' => $kelas->adminLms->email,
                'role' => 'admin',
                'tipe' => 'Admin Kelas',
            ]);
        }

        /**
         * ======================
         * FASILITATOR
         * ======================
         */
        if ($kelas->fasil) {
            $data->push((object)[
                'id'   => $kelas->fasil->id,
                'nama' => $kelas->fasil->name,
                'unit' => $kelas->fasil->email,
                'role' => 'fasil',
                'tipe' => 'Fasilitator',
            ]);
        }

        /**
         * ======================
         * PESERTA
         * ======================
         */
        foreach ($kelas->peserta as $p) {
            $data->push((object)[
                'id'   => $p->id,
                'nama' => $p->registrasi->nama ?? '-',
                'unit' => $p->registrasi->unit_kerja ?? '-',
                'role' => 'peserta',
                'tipe' => 'Peserta',
            ]);
        }

        /**
         * ======================
         * FILTER ROLE
         * ======================
         */
        if ($role !== 'all') {
            $data = $data->where('role', $role)->values();
        }

        /**
         * ======================
         * DATA MODAL
         * ======================
         */
        $pesertaTersedia = RegistrasiPeserta::where('pelatihan_id', $kelas->pelatihan_id)
            ->orderBy('nama')
            ->get();

        $adminUsers = User::orderBy('name')->get();

        return view('admin.kelas.show', compact(
            'kelas',
            'data',
            'pesertaTersedia',
            'adminUsers'
        ));
    }



    public function updatePengelola(Request $request, PelatihanKelas $kelas)
    {
        $data = $request->validate([
            'admin_lms_user_id' => ['nullable', 'exists:users,id'],
            'fasil_user_id'     => ['nullable', 'exists:users,id'],
        ]);

        $kelas->update($data);

        return back()->with('success', 'Pengelola kelas berhasil diperbarui.');
    }

    public function assignPeserta(Request $request, PelatihanKelas $kelas)
    {
        $data = $request->validate([
            'registrasi_peserta_id' => ['required', 'exists:registrasi_peserta,id'],
        ]);

        $registrasi = RegistrasiPeserta::findOrFail($data['registrasi_peserta_id']);

        // validasi pelatihan
        if ((int)$registrasi->pelatihan_id !== (int)$kelas->pelatihan_id) {
            return back()->with('error', 'Peserta tidak sesuai dengan pelatihan kelas ini.');
        }

        PelatihanPeserta::create([
            'pelatihan_id'        => $kelas->pelatihan_id,
            'pelatihan_kelas_id'  => $kelas->id,
            'registrasi_peserta_id' => $registrasi->id,
            'status_keikutsertaan' => 'terdaftar',
        ]);

        return back()->with('success', 'Peserta berhasil dimasukkan ke kelas.');
    }

    public function removePeserta(PelatihanKelas $kelas, PelatihanPeserta $peserta)
    {
        if ((int) $peserta->pelatihan_kelas_id !== (int) $kelas->id) {
            return back()->with('error', 'Peserta tidak berada di kelas ini.');
        }

        $peserta->pelatihan_kelas_id = null;
        $peserta->save();

        return back()->with('success', 'Peserta berhasil dikeluarkan dari kelas.');
    }

    public function store(Request $request, Pelatihan $pelatihan)
    {
        $data = $request->validate([
            'kode_kelas' => ['required', 'string', 'max:10'],
            'nama_kelas' => ['required', 'string', 'max:100'],
        ]);

        $exists = PelatihanKelas::where('pelatihan_id', $pelatihan->id)
            ->where('kode_kelas', $data['kode_kelas'])
            ->exists();

        if ($exists) {
            return redirect()
                ->route('pelatihan.kelola', $pelatihan->id)
                ->with('error', 'Kode kelas sudah dipakai di pelatihan ini.')
                ->withInput();
        }

        PelatihanKelas::create([
            'pelatihan_id' => $pelatihan->id,
            'kode_kelas'   => $data['kode_kelas'],
            'nama_kelas'   => $data['nama_kelas'],
        ]);

        return redirect()
            ->route('pelatihan.kelola', $pelatihan->id)
            ->with('success', 'Kelas berhasil ditambahkan.');
    }

    public function update(Request $request, Pelatihan $pelatihan, PelatihanKelas $kelas)
    {
        abort_if($kelas->pelatihan_id !== $pelatihan->id, 404);

        $data = $request->validate([
            'kode_kelas' => ['required', 'string', 'max:10'],
            'nama_kelas' => ['required', 'string', 'max:100'],
        ]);

        $exists = PelatihanKelas::where('pelatihan_id', $pelatihan->id)
            ->where('kode_kelas', $data['kode_kelas'])
            ->where('id', '!=', $kelas->id)
            ->exists();

        if ($exists) {
            return redirect()
                ->route('pelatihan.kelola', $pelatihan->id)
                ->with('error', 'Kode kelas sudah dipakai di pelatihan ini.')
                ->withInput();
        }

        $kelas->update($data);

        return redirect()
            ->route('pelatihan.kelola', $pelatihan->id)
            ->with('success', 'Kelas berhasil diperbarui.');
    }

    public function destroy(Pelatihan $pelatihan, PelatihanKelas $kelas)
    {
        abort_if($kelas->pelatihan_id !== $pelatihan->id, 404);

        // lepasin peserta biar aman
        PelatihanPeserta::where('pelatihan_kelas_id', $kelas->id)
            ->update(['pelatihan_kelas_id' => null]);

        $kelas->delete();

        return redirect()
            ->route('pelatihan.kelola', $pelatihan->id)
            ->with('success', 'Kelas berhasil dihapus.');
    }
}
