<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PelatihanKelas;
use App\Models\PelatihanKelasJadwal;
use Illuminate\Http\Request;

class PelatihanKelasJadwalController extends Controller
{
    public function index(PelatihanKelas $kelas)
    {
        $kelas->load('pelatihan');

        $jadwals = PelatihanKelasJadwal::where('pelatihan_kelas_id', $kelas->id)
            ->orderBy('tanggal')
            ->orderBy('jam_mulai')
            ->get();

        return view('admin.jadwal.index', compact('kelas', 'jadwals'));
    }

    public function create(PelatihanKelas $kelas)
    {
        $kelas->load('pelatihan');
        return view('admin.jadwal.create', compact('kelas'));
    }

    public function store(Request $request, PelatihanKelas $kelas)
    {
        $validated = $request->validate([
            'tanggal'        => ['required', 'date'],
            'jam_mulai'      => ['required', 'date_format:H:i'],
            'jam_selesai'    => ['required', 'date_format:H:i', 'after:jam_mulai'],

            // wajib karena di database mapel NOT NULL
            'mapel'          => ['required', 'string', 'max:150'],

            // optional
            'materi'         => ['nullable', 'string'],
            'tempat'         => ['nullable', 'string', 'max:255'],
            'ishoma_mulai'   => ['nullable', 'date_format:H:i'],
            'ishoma_selesai' => ['nullable', 'date_format:H:i', 'after:ishoma_mulai'],
        ]);

        $validated['pelatihan_kelas_id'] = $kelas->id;

        PelatihanKelasJadwal::create($validated);

        return redirect()
            ->route('admin.kelas.jadwal.index', $kelas->id)
            ->with('success', 'Jadwal berhasil ditambahkan.');
    }

    public function edit(PelatihanKelas $kelas, PelatihanKelasJadwal $jadwal)
    {
        $kelas->load('pelatihan');

        if ($jadwal->pelatihan_kelas_id !== $kelas->id) {
            abort(404);
        }

        return view('admin.jadwal.edit', compact('kelas', 'jadwal'));
    }

    public function update(Request $request, PelatihanKelas $kelas, PelatihanKelasJadwal $jadwal)
    {
        if ($jadwal->pelatihan_kelas_id !== $kelas->id) {
            abort(404);
        }

        $validated = $request->validate([
            'tanggal'        => ['required', 'date'],
            'jam_mulai'      => ['required', 'date_format:H:i'],
            'jam_selesai'    => ['required', 'date_format:H:i', 'after:jam_mulai'],

            'mapel'          => ['required', 'string', 'max:150'],
            'materi'         => ['nullable', 'string'],
            'tempat'         => ['nullable', 'string', 'max:255'],
            'ishoma_mulai'   => ['nullable', 'date_format:H:i'],
            'ishoma_selesai' => ['nullable', 'date_format:H:i', 'after:ishoma_mulai'],
        ]);

        $jadwal->update($validated);

        return redirect()
            ->route('admin.kelas.jadwal.index', $kelas->id)
            ->with('success', 'Jadwal berhasil diperbarui.');
    }

    public function destroy(PelatihanKelas $kelas, PelatihanKelasJadwal $jadwal)
    {
        if ($jadwal->pelatihan_kelas_id !== $kelas->id) {
            abort(404);
        }

        $jadwal->delete();

        return redirect()
            ->route('admin.kelas.jadwal.index', $kelas->id)
            ->with('success', 'Jadwal berhasil dihapus.');
    }

    public function printAbsensiPeserta(PelatihanKelas $kelas, PelatihanKelasJadwal $jadwal)
    {
        abort(404, 'Fitur absensi peserta belum dibuat.');
    }

    public function printAbsensiFasil(PelatihanKelas $kelas, PelatihanKelasJadwal $jadwal)
    {
        abort(404, 'Fitur absensi fasil belum dibuat.');
    }
}
