<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use App\Models\Guru;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AuthController extends Controller
{
    private function redirectAfterLogin()
    {
        $user = Auth::user();

        // pastiin relasi role kebaca (kalau belum keload)
        if ($user && !$user->relationLoaded('role')) {
            $user->load('role');
        }

        if ($user && $user->role && $user->role->name === 'admin') {
            return redirect()->route('admin.dashboard');
        }

        return redirect()->route('portal.dashboard'); // user
    }

    public function showLogin()
    {
        if (Auth::check()) {
            return $this->redirectAfterLogin();
        }

        return view('auth.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email'    => ['required', 'email'],
            'password' => ['required'],
        ]);

        $remember = $request->boolean('remember');

        if (Auth::attempt($credentials, $remember)) {
            $request->session()->regenerate();
            return $this->redirectAfterLogin();
        }

        return back()->withErrors([
            'email' => 'Email atau password salah.',
        ])->onlyInput('email');
    }

    public function showRegister()
    {
        if (Auth::check()) {
            return $this->redirectAfterLogin();
        }

        return view('auth.register');
    }

    public function register(Request $request)
    {
        $request->validate([
            'name'     => ['required', 'string', 'max:255'],
            'email'    => ['required', 'email', 'unique:users,email'],
            'password' => ['required', 'min:6', 'confirmed'],
        ]);

        // ✅ pastiin role "user" ada (kalau belum ada, dibuat)
        $roleUser = Role::firstOrCreate(
            ['name' => 'user'],
            ['label' => 'Peserta / User']
        );

        // ✅ password biarin cast "hashed" yang handle (jangan Hash::make)
        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'password' => $request->password,
            'role_id'  => $roleUser->id,
        ]);

        // optional: auto-link kalau email cocok guru
        $guru = Guru::where('email', $user->email)->first();
        if ($guru) {
            $user->guru_id = $guru->id;
            $user->save();
        }

        // ✅ setelah register, jangan auto-login → arahkan ke login
        return redirect()->route('login')
            ->with('success', 'Registrasi berhasil. Silakan login dengan akun yang baru dibuat.');
    }

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login');
    }
}
