<?php

namespace App\Http\Controllers;

use App\Models\Guru;
use App\Models\Sekolah;
use App\Exports\GuruExport;
use App\Imports\GuruImport;
use App\Models\BidangBahasa;
use Illuminate\Http\Request;
use App\Imports\SekolahImport;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class GuruController extends Controller
{
    public function index(Request $request)
    {
        $search    = $request->input('search');
        $kabupaten = $request->input('kabupaten');
        $bidangId  = $request->input('bidang_bahasa_id');

        $query = Guru::with(['sekolah', 'bidangBahasa'])
            ->orderBy('nama_guru');

        // Pencarian berdasarkan nama / NIK / instansi
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('nama_guru', 'like', "%{$search}%")
                    ->orWhere('nik', 'like', "%{$search}%")
                    ->orWhere('instansi', 'like', "%{$search}%");
            });
        }

        // Filter kota via tabel sekolah (relasi)
        if ($kabupaten) {
            $query->whereHas('sekolah', function ($q) use ($kabupaten) {
                $q->where('kabupaten', $kabupaten);
            });
        }

        // Filter bidang bahasa
        if ($bidangId) {
            $query->where('bidang_bahasa_id', $bidangId);
        }

        $guru = $query->paginate(20)->withQueryString();

        // Opsi kabupaten untuk dropdown ambil dari sekolah (supaya sinkron)
        $kabupatenOptions = Sekolah::select('kabupaten')
            ->whereNotNull('kabupaten')
            ->distinct()
            ->orderBy('kabupaten')
            ->pluck('kabupaten')
            ->toArray();

        // Opsi bidang (kalau kamu pakai tabel bidang_bahasa)
        $bidangList = \App\Models\BidangBahasa::orderBy('nama_bahasa')->get();

        return view('admin.guru.index', compact(
            'guru',
            'search',
            'kabupaten',
            'kabupatenOptions',
            'bidangList',
            'bidangId'
        ));
    }

    public function create()
    {
        $sekolahList = Sekolah::orderBy('nama')->get();

        return view('admin.guru.create', [
            'sekolahList' => $sekolahList
        ]);
    }

    public function edit(Guru $guru)
    {
        $sekolahList = Sekolah::orderBy('nama')->get();

        return view('admin.guru.edit', [
            'guru'        => $guru,
            'sekolahList' => $sekolahList
        ]);
    }

    public function store(Request $request)
    {
        $data = $this->validateData($request);

        if ($request->hasFile('foto')) {
            $data['foto'] = $request->file('foto')->store('guru', 'public');
        }

        Guru::create($data);

        return redirect()->route('guru.index')->with('success', 'Data PTK berhasil ditambahkan.');
    }

    public function update(Request $request, Guru $guru)
    {
        $data = $this->validateData($request);

        if ($request->hasFile('foto')) {
            // hapus foto lama
            if ($guru->foto && Storage::disk('public')->exists($guru->foto)) {
                Storage::disk('public')->delete($guru->foto);
            }

            $data['foto'] = $request->file('foto')->store('guru', 'public');
        }

        $guru->update($data);

        return redirect()->route('guru.index')->with('success', 'Data PTK berhasil diperbarui.');
    }

    protected function validateData(Request $request)
    {
        return $request->validate([
            'nik'                  => 'required|string|max:50',
            'nama_guru'            => 'required|string|max:255',
            'gelar'                => 'nullable|string|max:100',
            'nuptk'                => 'nullable|string|max:50',
            'nip'                  => 'nullable|string|max:50',
            'instansi'             => 'nullable|string|max:255',
            'jabatan'             => 'nullable|string|max:255',
            'kab_kota'             => 'nullable|string|max:255',
            'provinsi'             => 'nullable|string|max:255',
            'sekolah_id'           => 'nullable|integer|exists:sekolah,id',
            'pangkat_golongan'     => 'nullable|string|max:100',
            'tempat_lahir'         => 'nullable|string|max:100',
            'tanggal_lahir'        => 'nullable|date',
            'agama'                => 'nullable|string|max:50',
            'alamat_instansi'      => 'nullable|string',
            'alamat_rumah'         => 'nullable|string',
            'no_hp'                => 'nullable|string|max:50',
            'email'                => 'nullable|email',
            'pendidikan_terakhir'  => 'nullable|string|max:100',
            'prodi'                => 'nullable|string|max:150',
            'npwp'                 => 'nullable|string|max:50',
            'no_rekening'          => 'nullable|string|max:50',
            'bidang_bahasa_id'     => 'nullable|exists:bidang_bahasa,id',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ]);
    }

    public function destroy(Guru $guru)
    {
        if ($guru->foto && Storage::disk('public')->exists($guru->foto)) {
            Storage::disk('public')->delete($guru->foto);
        }


        $guru->delete();

        return redirect()
            ->route('guru.index')
            ->with('success', 'Data guru berhasil dihapus.');
    }

    public function import(Request $request)
    {
        $request->validate([

            'file' => 'required|mimes:xlsx,xls'
        ]);

        ini_set('memory_limit', '2048M');
        set_time_limit(0);

        try {
            Excel::import(new GuruImport, $request->file('file'));

            return redirect()
                ->route('guru.index')

                ->with('success', 'Import data guru / PTK berhasil diproses.');
        } catch (\Throwable $e) {
            return back()->with('error', 'Gagal import: ' . $e->getMessage());
        }
    }

    public function export()
    {
        $fileName = 'data_guru_ptk_' . now()->format('Ymd_His') . '.xlsx';

        return Excel::download(new GuruExport, $fileName);
    }

    public function importSekolah(Request $request)
    {
        $request->validate([
            'file' => ['required', 'file', 'mimes:xlsx,xls', 'max:5120'],
        ]);

        try {
            Excel::import(new SekolahImport, $request->file('file'));

            return redirect()
                ->route('sekolah.index')
                ->with('success', 'Import data sekolah dari Dapodik berhasil diproses.');
        } catch (\Throwable $e) {
            return redirect()
                ->route('sekolah.index')
                ->with('error', 'Gagal import data sekolah: ' . $e->getMessage());
        }
    }


    public function show(Guru $guru)
    {
        return view('admin.guru.show', compact('guru'));
    }

    public function select2(Request $request)
    {
        $search = $request->input('q');

        $query = Guru::select('id', 'nama_guru', 'nip', 'nik', 'nuptk', 'sekolah_id')
            ->with('sekolah:id,nama')
            ->orderBy('nama_guru');

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('nama_guru', 'like', "%{$search}%")
                    ->orWhere('nip', 'like', "%{$search}%")
                    ->orWhere('nik', 'like', "%{$search}%")
                    ->orWhere('nuptk', 'like', "%{$search}%");
            });
        }

        $data = $query->limit(30)->get();

        $result = $data->map(function ($g) {
            $identitas = $g->nip ?: ($g->nik ?: ($g->nuptk ?: null));

            return [
                'id' => $g->id,
                'text' => $g->nama_guru
                    . ($identitas ? " — {$identitas}" : '')
                    . ($g->sekolah ? " — " . ($g->sekolah->nama ?? '') : ''),
            ];
        });

        return response()->json(['results' => $result]);
    }
}
