<?php

namespace App\Http\Controllers;

use App\Models\Pelatihan;
use App\Models\PelatihanKelas;
use App\Models\PelatihanPeserta;
use Illuminate\Http\Request;

class PelatihanController extends Controller
{
    public function index(Request $request)
    {
        $search   = $request->input('search');
        $kategori = $request->input('kategori');
        $status   = $request->input('status');

        $pelatihan = Pelatihan::when($search, function ($q) use ($search) {
                $q->where('nama_pelatihan', 'like', "%{$search}%");
            })
            ->when($kategori, function ($q) use ($kategori) {
                $q->where('kategori', $kategori);
            })
            ->when($status, function ($q) use ($status) {
                $q->where('status', $status);
            })
            ->orderBy('tanggal_mulai', 'desc')
            ->paginate(10)
            ->withQueryString();

        $kategoriList = ['BK', 'Matematika', 'Tendik', 'PembelajaranMendalam'];

        return view('admin.pelatihan.index', compact(
            'pelatihan',
            'search',
            'kategori',
            'status',
            'kategoriList'
        ));
    }

    public function create()
    {
        $kategoriList = ['BK', 'Matematika', 'Tendik', 'PembelajaranMendalam'];

        return view('admin.pelatihan.create', compact('kategoriList'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'nama_pelatihan'  => ['required', 'string', 'max:255'],
            'kategori'        => ['nullable', 'string', 'max:100'],
            'tanggal_mulai'   => ['nullable', 'date'],
            'tanggal_selesai' => ['nullable', 'date', 'after_or_equal:tanggal_mulai'],
            'status'          => ['required', 'in:draft,berjalan,selesai'],
            'penyelenggara'   => ['nullable', 'string', 'max:255'],
            'deskripsi'       => ['nullable', 'string'],
        ]);

        $pelatihan = Pelatihan::create($data);

        return redirect()->route('pelatihan.index')
            ->with('success', 'Data pelatihan berhasil ditambahkan.');
    }

    public function edit(Pelatihan $pelatihan)
    {
        $kategoriList = ['BK', 'Matematika', 'Tendik', 'PembelajaranMendalam'];

        return view('admin.pelatihan.edit', compact('pelatihan', 'kategoriList'));
    }

    public function update(Request $request, Pelatihan $pelatihan)
    {
        $data = $request->validate([
            'nama_pelatihan'  => ['required', 'string', 'max:255'],
            'kategori'        => ['nullable', 'string', 'max:100'],
            'tanggal_mulai'   => ['nullable', 'date'],
            'tanggal_selesai' => ['nullable', 'date', 'after_or_equal:tanggal_mulai'],
            'status'          => ['required', 'in:draft,berjalan,selesai'],
            'penyelenggara'   => ['nullable', 'string', 'max:255'],
            'deskripsi'       => ['nullable', 'string'],
        ]);

        $pelatihan->update($data);

        return redirect()->route('pelatihan.index')
            ->with('success', 'Data pelatihan berhasil diperbarui.');
    }

    public function destroy(Pelatihan $pelatihan)
    {
        $pelatihan->delete();

        return redirect()->route('pelatihan.index')
            ->with('success', 'Data pelatihan berhasil dihapus.');
    }

    public function kelola(Pelatihan $pelatihan)
    {
        $tab = request('tab', 'kelas');

        $kelasList = PelatihanKelas::where('pelatihan_id', $pelatihan->id)
            ->orderBy('kode_kelas')
            ->get();

        $pesertaPage = PelatihanPeserta::with(['guru', 'kelas'])
            ->where('pelatihan_id', $pelatihan->id)
            ->latest()
            ->paginate(15);

        return view('admin.pelatihan.kelola', compact('pelatihan', 'tab', 'kelasList', 'pesertaPage'));
    }

    public function show(Pelatihan $pelatihan)
    {
        return redirect()->route('pelatihan.kelola', $pelatihan->id)->with('info', null);
    }

}
