<?php

namespace App\Http\Controllers;

use App\Models\PelatihanPeserta;
use App\Models\Guru;
use App\Models\Pelatihan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PelatihanPesertaController extends Controller
{
    public function index(Request $request)
    {
        $search      = $request->input('search');
        $pelatihanId = $request->input('pelatihan_id');
        $status      = $request->input('status_keikutsertaan');

        $query = PelatihanPeserta::with(['guru.sekolah', 'pelatihan']);

        if ($search) {
            $like = "%{$search}%";
            $query->whereHas('guru', function ($q) use ($like) {
                $q->where('nama_guru', 'like', $like)
                  ->orWhere('nik', 'like', $like);
            });
        }

        if ($pelatihanId) {
            $query->where('pelatihan_id', $pelatihanId);
        }

        if ($status) {
            $query->where('status_keikutsertaan', $status);
        }

        $peserta = $query->orderByDesc('created_at')
            ->paginate(20)
            ->withQueryString();

        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')->get();
        $statusOptions = [
            'terdaftar'    => 'Terdaftar',
            'lulus'        => 'Lulus',
            'tidak_lulus'  => 'Tidak lulus',
        ];

        return view('admin.peserta.index', compact(
            'peserta',
            'pelatihanList',
            'statusOptions',
            'search',
            'pelatihanId',
            'status'
        ));
    }

    public function create()
    {
        $peserta = new PelatihanPeserta();

        // dropdown kini via AJAX
        $guruList = [];

        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')->get();

        return view('admin.peserta.create', compact('peserta', 'guruList', 'pelatihanList'));
    }


    public function store(Request $request)
    {
        $data = $request->validate([
            'guru_id'              => 'required|exists:guru,id',
            'pelatihan_id'         => 'required|exists:pelatihan,id',
            'status_keikutsertaan' => 'nullable|in:terdaftar,lulus,tidak_lulus',
            'sertifikat'           => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        if (empty($data['status_keikutsertaan'])) {
            $data['status_keikutsertaan'] = 'terdaftar';
        }

        // upload sertifikat jika ada
        if ($request->hasFile('sertifikat')) {
            $path = $request->file('sertifikat')
                ->store('sertifikat', 'public');

            $data['sertifikat_path'] = $path;
        }

        PelatihanPeserta::create($data);

        return redirect()
            ->route('peserta.index')
            ->with('success', 'Peserta pelatihan berhasil ditambahkan.');
    }

    // EDIT FORM
    public function edit(PelatihanPeserta $peserta)
    {
        $guruList = []; // tidak load semua data guru

        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')->get();

        return view('admin.peserta.edit', compact('peserta', 'guruList', 'pelatihanList'));
    }


    // UPDATE DATA
    public function update(Request $request, PelatihanPeserta $peserta)
    {
        $validated = $request->validate([
            'guru_id'              => 'required|exists:guru,id',
            'pelatihan_id'         => 'required|exists:pelatihan,id',
            'status_keikutsertaan' => 'required|in:terdaftar,lulus,tidak_lulus',
            'kab_kota'             => 'nullable|string|max:255',
            'sertifikat'           => 'nullable|file|mimes:pdf,jpg,jpeg,png',
        ]);

        // handle upload sertifikat kalau ada
        if ($request->hasFile('sertifikat')) {
            $path = $request->file('sertifikat')->store('sertifikat', 'public');
            $validated['sertifikat_path'] = $path;
        }

        $peserta->update($validated);

        return redirect()
            ->route('peserta.index')
            ->with('success', 'Data peserta pelatihan berhasil diperbarui.');
    }

    public function destroy(PelatihanPeserta $peserta)
    {
        $peserta->delete();

        return redirect()
            ->route('peserta.index')
            ->with('success', 'Peserta pelatihan berhasil dihapus.');
    }

    // dipakai di route user untuk unduh sertifikat
    public function downloadSertifikat(PelatihanPeserta $peserta)
    {
        if (! $peserta->sertifikat_path || $peserta->status_keikutsertaan !== 'lulus') {
            abort(404, 'Sertifikat belum tersedia.');
        }

        $disk = Storage::disk('public');

        if (! $disk->exists($peserta->sertifikat_path)) {
            abort(404, 'File sertifikat tidak ditemukan.');
        }

        return $disk->download($peserta->sertifikat_path, 'sertifikat-'.$peserta->id.'.pdf');
    }
}
